﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Dynamic;
using System.Text;
using System.Data;
using System.Data.Linq;
using VeteransAffairs.Registries.Business;

namespace VeteransAffairs.Registries.BusinessManager.TBI
{
    [Serializable()]
    [System.ComponentModel.DataObject]
    public class PolyTraumaEntriesManager : BaseBO
    {
        private UserAccountManager _user = (UserAccountManager)System.Threading.Thread.CurrentPrincipal;
        private RegistriesCommonManager _commonManager = new RegistriesCommonManager();

        public PolyTraumaEntriesManager()
        {
            _defaultSortField = "TBI_POLYTRAUMA_ID";
        }

        private IQueryable<TBI_POLYTRAUMA> LinqAll(RegistriesDataAccess _db)
        {
            //populate LinqAll
            IQueryable<TBI_POLYTRAUMA> tempLinq = (from e in _db.TBI_POLYTRAUMAs
                                                   select e);

            //add all business filtering rules 
            if (!_user.IsSuperUser)
            {
                //rule: add filter to return only items within the list of institutions accessible by the user
                int[] accessibleInstitutions = (from a in
                                                    _commonManager.GetActiveUserRoles(_user.RegistryId, _user.UserId)
                                                select a.STD_INSTITUTION_ID).Distinct().ToArray();

                //TO DO : Move the IQueryable calls above to userAccount Manager

                tempLinq = (from e in tempLinq
                            //join r in activeUserRoles on e.POLYTRAUMA_CENTER_ID equals r.STD_INSTITUTION_ID
                            where accessibleInstitutions.Contains(e.POLYTRAUMA_CENTER_ID.GetValueOrDefault())
                            select e).Distinct();
            }
            //TO DO : need to get this to filter on role properly

            return tempLinq;
        }

        private void SetLoadWith(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<TBI_POLYTRAUMA>(e => e.TBI_INJURY_DETAILs);
            lo.AssociateWith<TBI_POLYTRAUMA>(e => e.TBI_INJURY_DETAILs.Where(det => det.STD_TBIINJURYLIST.INACTIVE_FLAG == false));
            lo.LoadWith<TBI_INJURY_DETAIL>(e => e.STD_TBIINJURYLIST);
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.TBI_FOLLOWUPs);
            lo.LoadWith<TBI_FOLLOWUP>(e => e.STD_TBIIFOLLOWUPLIST);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        private void SetLoadWithForListViews(RegistriesDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();

            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_INSTITUTION);  //loads Institution for PATIENT_HOME_VAMC_ID
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_INSTITUTION1); //loads Institution for POLYTRAUMA_CENTER_ID
            lo.LoadWith<STD_INSTITUTION>(e => e.STD_INSTITUTION_PARENT); //loads parent institution of any institution in order to get the VISN
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_TBIPROVIDERLOC);
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.STD_TBIFACILITYSRC);
            lo.LoadWith<TBI_POLYTRAUMA>(e => e.PATIENT);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #region Method for Check Existence

        public bool PolyTraumaExists(int id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.TBI_POLYTRAUMAs where e.TBI_POLYTRAUMA_ID == id select e).Count();

                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        public bool SSNExistsInPolyTrauma(string Snum)
        {
            using (_db = GetDataContext())
            {

                int count = (from e in _db.TBI_POLYTRAUMAs where e.Snum.Replace("-", "") == Snum.Replace("-", "") select e).Count();

                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        #endregion

        #region Methods for Select

        /// <summary>
        /// Select Method to return IQueryable of TBI_POLYTRAUMA with user access logic in place
        /// </summary>
        /// <returns>IQueryable of TBI_POLYTRAUMA</returns>
        public IQueryable<TBI_POLYTRAUMA> SelectIQueryable(RegistriesDataAccess _db)
        {
            //SetLoadWithForListViews(_db);
            IQueryable<TBI_POLYTRAUMA> entities;
            entities = SelectLinqFilter(_db);

            return entities;

        }

        /// <summary>
        /// Select Method to return IEnumerable of TBI_POLYTRAUMA with user access logic in place
        /// </summary>
        /// <returns>IEnumerable of TBI_POLYTRAUMA</returns>
        public IEnumerable<TBI_POLYTRAUMA> Select(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                List<TBI_POLYTRAUMA> entities;
                entities = SelectLinqFilter(_db).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        public int SelectCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectLinqFilter(_db).Count();
            }
        }

        private IQueryable<TBI_POLYTRAUMA> SelectLinqFilter(RegistriesDataAccess _db)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);
            linqFilter = from t in linqFilter where t.INACTIVE_FLAG == false select t;
            return linqFilter;
        }

        public IEnumerable<TBI_POLYTRAUMA> SelectInactive(string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }

            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                List<TBI_POLYTRAUMA> entities;
                entities = SelectInactiveLinqFilter(_db).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();

                return entities;
            }
        }

        public int SelectInactiveCount(string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectInactiveLinqFilter(_db).Count();
            }
        }

        private IQueryable<TBI_POLYTRAUMA> SelectInactiveLinqFilter(RegistriesDataAccess _db)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);
            linqFilter = from t in linqFilter where t.INACTIVE_FLAG == true select t;
            return linqFilter;
        }

        public bool IsDischargeToActive(int? id)
        {
            using (_db = GetDataContext())
            {
                int count = (from e in _db.STD_TBIDISCHARGETOs
                             where e.ID == id && e.INACTIVE_FLAG != true
                             select e).Count();
                if (count > 0)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        public string GetDischargeToName(int? id)
        {
            string name = string.Empty;
            using (_db = GetDataContext())
            {
                name = (from e in _db.STD_TBIDISCHARGETOs
                        where e.ID == id
                        select e.NAME).SingleOrDefault();
            }
            return name;
        }

        #endregion

        #region Methods for Select By PolyTraum VAMC

        public IEnumerable<TBI_POLYTRAUMA> SelectByPolyTraumaVAMC(int polyTraumaVamcID, int entryTypeID, string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = _defaultSortField;
            }
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                List<TBI_POLYTRAUMA> entities;
                entities = SelectByPolyTraumaVAMCLinqFilter(polyTraumaVamcID, entryTypeID).OrderBy(sort).Skip(startRow).Take(maxRows).ToList();
                return entities;
            }
        }

        public int SelectByPolyTraumaVAMCCount(int polyTraumaVamcID, int entryTypeID, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                SetLoadWithForListViews(_db);
                return SelectByPolyTraumaVAMCLinqFilter(polyTraumaVamcID, entryTypeID).Count();
            }
        }

        private IQueryable<TBI_POLYTRAUMA> SelectByPolyTraumaVAMCLinqFilter(int polyTraumaVamcID, int entryTypeID)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);
            linqFilter = from t in linqFilter where t.INACTIVE_FLAG == false select t;

            if (polyTraumaVamcID != 0)
            {
                linqFilter = from t in linqFilter where t.POLYTRAUMA_CENTER_ID == polyTraumaVamcID select t;
            }
            if (entryTypeID != 0)
            {
                //use bitwise operator to denote that 1 and 3 should be returned when 1 is selected
                //and that 2 and 3 should be returned when 2 is selected. If new codes are added then
                //this code will have to be modified. This is not expected.
                linqFilter = from t in linqFilter where (t.ENTRY_TYPE_ID & entryTypeID) == entryTypeID select t;
            }
            return linqFilter;
        }

        #endregion

        #region Methods for Select Download View By PolyTraum VAMC

        public IEnumerable<PolyTraumaDownloadView> SelectDownLoadViewByPolyTraumaVAMC(int polyTraumaVamcID, int entryTypeID, string sort, int startRow, int maxRows)
        {
            if (string.IsNullOrEmpty(sort))
            {
                sort = "ID";
            }

            using (_db = GetDataContext())
            {
                SetLoadWith(_db);

                IEnumerable<PolyTraumaDownloadView> entities;
                entities =

                (from e in SelectDownLoadViewByPolyTraumaVAMCLinqFilter(polyTraumaVamcID, entryTypeID)

                 select new PolyTraumaDownloadView
                 {
                     ID = e.TBI_POLYTRAUMA_ID,
                     SSN = e.Snum,
                     PatientName = e.FullName,
                     MTFReferralSource = e.STD_TBIFACILITYSRC.NAME,
                     InjuryDate = getShortDateStringFromNullableDate(e.TBI_INJURY_DATE),
                     Secondary_InjuryDate = getShortDateStringFromNullableDate(e.SECONDARY_INJURY_DATE),
                     Tertiary_InjuryDate = getShortDateStringFromNullableDate(e.TERTIARY_INJURY_DATE),
                     Quarternary_InjuryDate = getShortDateStringFromNullableDate(e.QUINARY_INJURY_DATE),
                     Quinary_InjuryDate = getShortDateStringFromNullableDate(e.QUATERNARY_INJURY_DATE),
                     InjuryGeographicLocation = e.STD_COMBATLOCATION.NAME,
                     BlastExplosion = getStringFromNullableChar(e.BLAST_EXPLOSION_INJRY_IND),
                     VehicleInjury = getStringFromNullableChar(e.VEHICLE_INJRY_IND),
                     BulletInjury = getStringFromNullableChar(e.BULLET_INJRY_IND),
                     OtherInjury = getStringFromNullableChar(e.OTHER_INJRY_IND),
                     OtherInjuryDescription = e.OTHER_INJRY_DSCR,
                     InjuryDetailList = e.TBI_INJURY_DETAILs,
                     ProviderLocation = e.PROVIDER_LOCATION,
                     FollowupPolyTraumaCaseManager = e.PROVIDER_NAME,
                     ProviderEmail = e.PROVIDER_EMAIL,
                     ProviderPhone = e.PROVIDER_PHONE,
                     PolyTraumVAMC = e.STD_INSTITUTION1.NAME,
                     IsEmergingConsciousness = (e.EMERGE_CONSCIOUSNESS_FLAG == 1 ? true : false),
                     DetailedEmergingConsciousnessEtiology = e.STD_TBIEMERGINGETIOLOGY.NAME,
                     DateOfAlteredConsciousness = getShortDateStringFromNullableDate(e.ALTERED_CONSCIOUSNESS_DATE),
                     DatePatientEmerged = getShortDateStringFromNullableDate(e.EMERGE_CONSCIOUSNESS_DATE),
                     AdmissionDate = getShortDateStringFromNullableDate(e.REHAB_ADMIT_DATE),
                     NumberOfDaysOffUnit = e.DAYS_OFF_UNIT_TOTAL.ToString(),
                     AdmissionFIMCode = e.ADMISSION_FIM_CODE,
                     FRGCode = e.STD_TBI_FRG.CODE,
                     RehabDischargeDate = getShortDateStringFromNullableDate(e.REHAB_DISCHARGE_DATE),
                     DischargeFIMCode = e.DISCHARGE_FIM_CODE,
                     TBIDischargedTo = e.STD_TBIDISCHARGETO3.NAME,
                     FollowupList = e.TBI_FOLLOWUPs,
                     FollowupSiteDescription = e.FOLLOWUP_SITE_DSCR,
                     LOS = e.TotalLengthOfStay.ToString(),
                     AgeAtAdmission = e.AgeAtAdmission.ToString(),
                     PatientBirthDate = getShortDateStringFromNullableDate(e.BIRTH_DATE),
                     PatientDeathDate = getShortDateStringFromNullableDate(e.DEATH_DATE),
                     PatientGender = e.STD_GENDER.NAME,
                     PatientMaritalStatus = e.STD_MARITALSTATUS.NAME,
                     PatientRace = e.STD_RACE.NAME,
                     PatientEthnicity = e.STD_ETHNICITY.NAME,
                     PatientAddress1 = e.ADDRESS_LINE1,
                     PatientAddress2 = e.ADDRESS_LINE2,
                     PatientAddress3 = e.ADDRESS_LINE3,
                     PatientCity = e.CITY,
                     PatientState = e.STATE,
                     PatientCounty = e.COUNTY,
                     PatientCountry = e.COUNTRY,
                     PatientPostalCode = e.POSTAL_CODE,
                     PatientHomePhone = e.HOME_PHONE,
                     PatientEmail = e.EMAIL_ADDRESS,
                     PatientHomeVAMC = e.STD_INSTITUTION.NAME,
                     PatientServiceBranch = e.STD_SERVICEBRANCH.NAME,

                     //PatientMilitaryDutyStatus=e.STD_MILITARYDUTYSTS.NAME
                     
                     PTRPMilitaryDutyStatus_Admission  = e.STD_MILITARYDUTYST.NAME,
                     PTRPMilitaryDutyStatus_Discharge=getDischargeStsNameByID(e.MilitaryDutyStatus_Discharge.ToString()),

                     PRCMilitaryDutyStatus_Admission = getMilDutyStsNameByID(e.PRCMilitaryDutyStatus_Admission.ToString()),
                     PRCMilitaryDutyStatus_Discharge= getDischargeStsNameByID(e.PRCMilitaryDutyStatus_Discharge.ToString()),
                     
                     //MilitaryStatusAtDischarge = e.STD_TBIDSCHRGST.NAME,
                     RecordCreatedOn = e.CREATED,
                     RecordCreatedBy = e.CREATEDBY,
                     RecordLastUpdatedOn = e.UPDATED,
                     RecordLastUpdatedBy = e.UPDATEDBY,

                     //TLB: Added 3/5/2015 Per RTC 159357
                     ENTRY_TYPE = e.STD_POLYTRAUMA_ENTRY_TYPE.NAME,
                     IS_FOLLOW_UP = (e.IS_FOLLOW_UP.HasValue ? e.IS_FOLLOW_UP.Value : false),
                     JFK_COMA_RECOVERY = e.JFK_COMA_RECOVERY.Value.ToString(),
                     CASE_MANAGER_NAME = e.CASE_MANAGER_NAME,
                     DAYS_FROM_ONSET = e.DAYS_FROM_ONSET.Value.ToString(),
                     GENITO_URINARY_INJURIES = (e.GENITO_URINARY_INJURIES.HasValue ? e.GENITO_URINARY_INJURIES.Value : false),
                     PRC_FOLLOWUP_DATE = getShortDateStringFromNullableDate(e.PRC_FOLLOWUP_DATE),
                     PHYS_ASSAULT_INJRY_IND = getStringFromNullableChar(e.PHYS_ASSAULT_INJRY_IND),
                     PTRP_PROGRAM = e.PTRP_PROGRAM,
                     PTRP_ADMISSION_DATE = getShortDateStringFromNullableDate(e.PTRP_ADMIT_DATE),
                     PTRP_DISCHARGE_DATE = getShortDateStringFromNullableDate(e.PTRP_DISCHARGE_DATE),
                     PTRP_FOLLOWUP_DATE = getShortDateStringFromNullableDate(e.PTRP_FOLLOWUP_DATE),
                     SEVERITY_LEVEL_INDEX = e.STD_POLYTRAUMA_SEVERITY_LEVEL_INDEX.NAME,
                     PREINJURY_RESIDENCE = e.STD_TBIDISCHARGETO2.NAME,
                     DC_RESIDENCE = e.STD_TBIDISCHARGETO.NAME,
                     FOLLOWUP_RESIDENCE = e.STD_TBIDISCHARGETO1.NAME,
                     PREINJURY_PRIM_PERSON_LIVING_WITH = e.STD_PERSON_LIVING_WITH.NAME,
                     PRI_PERSON_LIVING_DC = e.STD_PERSON_LIVING_WITH1.NAME,
                     PRI_PERSON_LIVING_FOLLOWUP = e.STD_PERSON_LIVING_WITH2.NAME,
                     PRI_PREINJURY_EMPLOY_STATUS = e.STD_POLYTRAUMA_EMPLOYMENT_STATUS2.NAME,
                     PRI_DC_EMPLOY_STATUS = e.STD_POLYTRAUMA_EMPLOYMENT_STATUS.NAME,
                     PRI_FOLLOWUP_EMPLOY_STATUS = e.STD_POLYTRAUMA_EMPLOYMENT_STATUS1.NAME,
                     SEC_PREINJURY_EMPLOY_STATUS = e.STD_POLYTRAUMA_EMPLOYMENT_STATUS5.NAME,
                     SEC_DC_EMPLOY_STATUS = e.STD_POLYTRAUMA_EMPLOYMENT_STATUS3.NAME,
                     SEC_FOLLOWUP_EMPLOY_STATUS = e.STD_POLYTRAUMA_EMPLOYMENT_STATUS4.NAME,
                     MPAI_ABILITIES_INITIAL = e.MPAI_ABILITIES_INITIAL.Value.ToString(),
                     MPAI_ABILITIES_DC = e.MPAI_ABILITIES_DC.Value.ToString(),
                     MPAI_ABILITIES_FOLLOWUP = e.MPAI_ABILITIES_FOLLOWUP.Value.ToString(),
                     MPAI_ADJUSTMENT_INITIAL = e.MPAI_ADJUSTMENT_INITIAL.Value.ToString(),
                     MPAI_ADJUSTMENT_DC = e.MPAI_ADJUSTMENT_DC.Value.ToString(),
                     MPAI_ADJUSTMENT_FOLLOWUP = e.MPAI_ADJUSTMENT_FOLLOWUP.Value.ToString(),
                     MPAI_PARTICIPATION_INITIAL = e.MPAI_PARTICIPATION_INITIAL.Value.ToString(),
                     MPAI_PARTICIPATION_DC = e.MPAI_PARTICIPATION_DC.Value.ToString(),
                     MPAI_PARTICIPATION_FOLLOWUP = e.MPAI_PARTICIPATION_FOLLOWUP.Value.ToString(),
                     MPAI_TOTAL_INITIAL = e.MPAI_TOTAL_INITIAL.Value.ToString(),
                     MPAI_TOTAL_DC = e.MPAI_TOTAL_DC.Value.ToString(),
                     MPAI_TOTAL_FOLLOWUP = e.MPAI_TOTAL_FOLLOWUP.Value.ToString(),
                     MPAI_Q26_ADMISSION = e.MPAI_Q26_ADMISSION.Value.ToString(),
                     MPAI_Q26_DC = e.MPAI_Q26_DC.Value.ToString(),
                     MPAI_Q26_FOLLOWUP = e.MPAI_Q26_FOLLOWUP.Value.ToString(),
                     PART_FOLLOWUP = e.PART_FOLLOWUP.Value.ToString(),
                     SATIS_W_LIFE_ADMISSION = e.SATIS_W_LIFE_ADMISSION.Value.ToString(),
                     SATIS_W_LIFE_DC = e.SATIS_W_LIFE_DC.Value.ToString(),
                     SATIS_W_LIFE_FOLLOWUP = e.SATIS_W_LIFE_FOLLOWUP.Value.ToString(),
                     SECONDARY_INJURY_DATE = getShortDateStringFromNullableDate(e.SECONDARY_INJURY_DATE),
                     TERTIARY_INJURY_DATE = getShortDateStringFromNullableDate(e.TERTIARY_INJURY_DATE),
                     QUATERNARY_INJURY_DATE = getShortDateStringFromNullableDate(e.QUATERNARY_INJURY_DATE),
                     QUINARY_INJURY_DATE = getShortDateStringFromNullableDate(e.QUINARY_INJURY_DATE)

                 }).OrderBy(sort).ToList();

                return entities;

            }
        }
        private String getDischargeStsNameByID(String ID)
        {

            String _sName = "";
            if (!String.IsNullOrEmpty(ID))
            {
                if (Convert.ToInt32(ID) > 0)
                {
                    int mID = Convert.ToInt32(ID);
                    using (_db = GetDataContext())
                    {

                        _sName = (from e in _db.STD_TBIDSCHRGSTs
                                  where e.ID == mID
                                  select e.NAME).FirstOrDefault().ToString();
                    }
                }
            }
            return _sName;
        }
        private String getMilDutyStsNameByID(String ID)
        {
            
            String _sName = "";
            if (!String.IsNullOrEmpty(ID))
            {
                if (Convert.ToInt32(ID) > 0)
                {
                    int mID=Convert.ToInt32(ID);
                    using (_db = GetDataContext())
                    {

                       _sName= (from e in _db.STD_MILITARYDUTYSTs
                         where e.ID == mID
                         select e.NAME).FirstOrDefault().ToString();
                    }
                }
            }
            return _sName;
        }

        public int SelectDownLoadViewByPolyTraumaVAMCCount(int polyTraumaVamcID, int entryTypeID, string sort, int startRow, int maxRows)
        {
            using (_db = GetDataContext())
            {
                //SetLoadWith(_db);
                return SelectDownLoadViewByPolyTraumaVAMCLinqFilter(polyTraumaVamcID, entryTypeID).Count();
            }
        }

        private IQueryable<TBI_POLYTRAUMA> SelectDownLoadViewByPolyTraumaVAMCLinqFilter(int polyTraumaVamcID, int entryTypeID)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);
            linqFilter = from t in linqFilter where t.INACTIVE_FLAG == false select t;

            if (polyTraumaVamcID != 0)
            {
                linqFilter = from t in linqFilter where t.POLYTRAUMA_CENTER_ID == polyTraumaVamcID select t;
            }
            if (entryTypeID != 0)
            {
                linqFilter = from t in linqFilter where t.ENTRY_TYPE_ID == entryTypeID select t;
            }
            return linqFilter;
        }

        private string getShortDateStringFromNullableDate(DateTime? inDate)
        {
            string returnString = "";
            if (inDate.HasValue)
            {
                returnString = Convert.ToDateTime(inDate).ToShortDateString();
            }
            return returnString;
        }

        private string getStringFromNullableChar(char? inChar)
        {
            string returnString = "";
            if (inChar.HasValue)
            {
                returnString = inChar.Value.ToString();
            }
            return returnString;
        }

        public class PolyTraumaDownloadView
        {
            public int ID { get; set; }
            public string SSN { get; set; }
            public string PatientName { get; set; }
            public string MTFReferralSource { get; set; }
            public string InjuryDate { get; set; }
            public string Secondary_InjuryDate { get; set; }
            public string Tertiary_InjuryDate { get; set; }
            public string Quarternary_InjuryDate { get; set; }
            public string Quinary_InjuryDate { get; set; }
            public string InjuryGeographicLocation { get; set; }
            public string BlastExplosion { get; set; }
            public string VehicleInjury { get; set; }
            public string BulletInjury { get; set; }
            public string OtherInjury { get; set; }
            public string OtherInjuryDescription { get; set; }
            public IEnumerable<TBI_INJURY_DETAIL> InjuryDetailList { get; set; }
            public string InjuryDetails
            {
                get
                {
                    string _temp = "";

                    if (InjuryDetailList.Count() > 0)
                    {
                        _temp = InjuryDetailList.Select(e => e.STD_TBIINJURYLIST.NAME.ToString()).Aggregate((i, next) => i + ", " + next);
                    }

                    return _temp;
                }
            }
            public string ProviderLocation { get; set; }
            public string FollowupPolyTraumaCaseManager { get; set; }
            public string ProviderEmail { get; set; }
            public string ProviderPhone { get; set; }
            public string PolyTraumVAMC { get; set; }
            public bool IsEmergingConsciousness { get; set; }
            public string DetailedEmergingConsciousnessEtiology { get; set; }
            public string DateOfAlteredConsciousness { get; set; }
            public string DatePatientEmerged { get; set; }
            public string AdmissionDate { get; set; }
            public string NumberOfDaysOffUnit { get; set; }
            public string AdmissionFIMCode { get; set; }
            public string FRGCode { get; set; }
            public string RehabDischargeDate { get; set; }
            public string DischargeFIMCode { get; set; }
            public string TBIDischargedTo { get; set; }
            public IEnumerable<TBI_FOLLOWUP> FollowupList { get; set; }
            public string FollowupClinicalNeeds
            {
                get
                {
                    string _temp = "";

                    if (FollowupList.Count() > 0)
                    {
                        _temp = FollowupList.Select(e => e.STD_TBIIFOLLOWUPLIST.NAME.ToString()).Aggregate((i, next) => i + ", " + next);
                    }

                    return _temp;
                }
            }
            public string FollowupSiteDescription { get; set; }
            public string LOS { get; set; }
            public string AgeAtAdmission { get; set; }
            public string PatientBirthDate { get; set; }
            public string PatientDeathDate { get; set; }
            public string PatientGender { get; set; }
            public string PatientMaritalStatus { get; set; }
            public string PatientRace { get; set; }
            public string PatientEthnicity { get; set; }
            public string PatientAddress1 { get; set; }
            public string PatientAddress2 { get; set; }
            public string PatientAddress3 { get; set; }
            public string PatientCity { get; set; }
            public string PatientState { get; set; }
            public string PatientCounty { get; set; }
            public string PatientCountry { get; set; }
            public string PatientPostalCode { get; set; }
            public string PatientHomePhone { get; set; }
            public string PatientEmail { get; set; }
            public string PatientHomeVAMC { get; set; }
            public string PatientServiceBranch { get; set; }
            public string PTRPMilitaryDutyStatus_Admission { get; set; }
            public string PTRPMilitaryDutyStatus_Discharge { get; set; }
            public string PRCMilitaryDutyStatus_Admission { get; set; }
            public string PRCMilitaryDutyStatus_Discharge { get; set; }
            //public string MilitaryStatusAtDischarge { get; set; }
            public DateTime RecordCreatedOn { get; set; }
            public string RecordCreatedBy { get; set; }
            public DateTime RecordLastUpdatedOn { get; set; }
            public string RecordLastUpdatedBy { get; set; }

            //TLB: Added 3/5/2015 Per RTC 159357
            public string ENTRY_TYPE { get; set; }
            public bool IS_FOLLOW_UP { get; set; }
            public string JFK_COMA_RECOVERY { get; set; }
            public string CASE_MANAGER_NAME { get; set; }
            public string DAYS_FROM_ONSET { get; set; }
            public bool GENITO_URINARY_INJURIES { get; set; }
            public string PRC_FOLLOWUP_DATE { get; set; }
            public string PHYS_ASSAULT_INJRY_IND { get; set; }
            public string PTRP_PROGRAM { get; set; }
            public string PTRP_ADMISSION_DATE { get; set; }
            public string PTRP_DISCHARGE_DATE { get; set; }
            public string PTRP_FOLLOWUP_DATE { get; set; }
            public string SEVERITY_LEVEL_INDEX { get; set; }
            public string PREINJURY_RESIDENCE { get; set; }
            public string DC_RESIDENCE { get; set; }
            public string FOLLOWUP_RESIDENCE { get; set; }
            public string PREINJURY_PRIM_PERSON_LIVING_WITH { get; set; }
            public string PRI_PERSON_LIVING_DC { get; set; }
            public string PRI_PERSON_LIVING_FOLLOWUP { get; set; }
            public string PRI_PREINJURY_EMPLOY_STATUS { get; set; }
            public string PRI_DC_EMPLOY_STATUS { get; set; }
            public string PRI_FOLLOWUP_EMPLOY_STATUS { get; set; }
            public string SEC_PREINJURY_EMPLOY_STATUS { get; set; }
            public string SEC_DC_EMPLOY_STATUS { get; set; }
            public string SEC_FOLLOWUP_EMPLOY_STATUS { get; set; }
            public string MPAI_ABILITIES_INITIAL { get; set; }
            public string MPAI_ABILITIES_DC { get; set; }
            public string MPAI_ABILITIES_FOLLOWUP { get; set; }
            public string MPAI_ADJUSTMENT_INITIAL { get; set; }
            public string MPAI_ADJUSTMENT_DC { get; set; }
            public string MPAI_ADJUSTMENT_FOLLOWUP { get; set; }
            public string MPAI_PARTICIPATION_INITIAL { get; set; }
            public string MPAI_PARTICIPATION_DC { get; set; }
            public string MPAI_PARTICIPATION_FOLLOWUP { get; set; }
            public string MPAI_TOTAL_INITIAL { get; set; }
            public string MPAI_TOTAL_DC { get; set; }
            public string MPAI_TOTAL_FOLLOWUP { get; set; }
            public string MPAI_Q26_ADMISSION { get; set; }
            public string MPAI_Q26_DC { get; set; }
            public string MPAI_Q26_FOLLOWUP { get; set; }
            public string PART_FOLLOWUP { get; set; }
            public string SATIS_W_LIFE_ADMISSION { get; set; }
            public string SATIS_W_LIFE_DC { get; set; }
            public string SATIS_W_LIFE_FOLLOWUP { get; set; }
            public string SECONDARY_INJURY_DATE { get; set; }
            public string TERTIARY_INJURY_DATE { get; set; }
            public string QUATERNARY_INJURY_DATE { get; set; }
            public string QUINARY_INJURY_DATE { get; set; }
        }

        #endregion

        #region Methods for SelectByID

        public TBI_POLYTRAUMA SelectByID(int id)
        {
            using (_db = GetDataContext())
            {
                SetLoadWith(_db);
                TBI_POLYTRAUMA polyTrauma;

                if (id == 0)
                {
                    polyTrauma = new TBI_POLYTRAUMA();
                    _db.TBI_POLYTRAUMAs.Attach(polyTrauma);

                    polyTrauma.SetAsChangeTrackingRoot(VeteransAffairs.Registries.Business.EntityState.New, true);
                    polyTrauma.SetAsInsertOnSubmit();
                }
                else
                {
                    polyTrauma = SelectByIDLinqFilter(id).SingleOrDefault();

                    if (polyTrauma != null)
                    {
                        polyTrauma.SetAsChangeTrackingRoot(true);
                    }

                }

                if (polyTrauma != null)
                {
                    int status = polyTrauma.EnforceBusinessRules(true, _db);
                }

                return polyTrauma;

            }

        }

        private IQueryable<TBI_POLYTRAUMA> SelectByIDLinqFilter(int id)
        {
            IQueryable<TBI_POLYTRAUMA> linqFilter = LinqAll(_db);

            linqFilter = from t in linqFilter where t.TBI_POLYTRAUMA_ID == id select t;

            return linqFilter;
        }

        #endregion

        #region Methods for Update

        /// <summary>
        /// Update Method.  Returns an int representing the ID of the item if the update/insert was sucessful.
        /// </summary>
        public int Update(TBI_POLYTRAUMA polyTrauma)
        {
            int returnValue = 0;
            using (_db = GetDataContext())
            {
                _db.DeferredLoadingEnabled = false;
                polyTrauma.EnforceBusinessRules(true, _db);

                //create an instance of the custom eventArgs in order to populate the id selected
                BOSaveSuccessEventArgs eventArgs = new BOSaveSuccessEventArgs();

                if (polyTrauma.BrokenRules.Count == 0)
                {
                    eventArgs.SaveStatusArg = SaveStatus.SaveFail;

                    //this line traverses all entities, attaching all of them as appropriate to the data context.
                    polyTrauma.SynchroniseWithDataContext(_db);
                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                        returnValue =
                        eventArgs.SavedItemId = polyTrauma.TBI_POLYTRAUMA_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                        returnValue =
                        eventArgs.SavedItemId = polyTrauma.TBI_POLYTRAUMA_ID;
                        eventArgs.SaveStatusArg = SaveStatus.SaveSuccess;
                    }
                    catch (Exception e)
                    {
                        eventArgs.SaveStatusArg = SaveStatus.SaveFail;
                    }
                }
                RaiseSaveEvent(this, eventArgs);
                return returnValue;
            }
        }


        public int Delete(int polyTraumaId)
        {
            int returnStatus = -1;
            if (polyTraumaId > 0)
            {
                using (_db = GetDataContext())
                {
                    TBI_POLYTRAUMA tbiPolytrauma = (from e in _db.TBI_POLYTRAUMAs
                                                    where e.TBI_POLYTRAUMA_ID == polyTraumaId
                                                    select e).FirstOrDefault();
                    _db.TBI_POLYTRAUMAs.DeleteOnSubmit(tbiPolytrauma);

                    try
                    {
                        _db.SubmitChanges(ConflictMode.ContinueOnConflict);
                        returnStatus = 0;
                    }
                    catch (ChangeConflictException)
                    {
                        _db.ChangeConflicts.ResolveAll(RefreshMode.KeepChanges);
                        returnStatus = 0;
                    }
                    catch
                    {
                        returnStatus = -1;
                    }
                }
            }
            return returnStatus;
        }

        #endregion

        #region Methods to change foreign key references

        /// <summary>
        /// Method to return Full Visn name of the PATIENT_HOME_VAMC.
        /// </summary>
        public string getHomeVAMCName(int polyTraumaId)
        {
            string homeVAMCName = "";

            if (polyTraumaId > 0)
            {
                using (_db = GetDataContext())
                {
                    homeVAMCName = (from e in _db.TBI_POLYTRAUMAs
                                    where e.TBI_POLYTRAUMA_ID == polyTraumaId
                                    select e.STD_INSTITUTION.FullVISNName).FirstOrDefault().ToString();
                }
            }
            return homeVAMCName;
        }

        /// <summary>
        /// Method to return a new selected patient (PATIENT) in order to modify the linq entity.
        /// </summary>
        public TBI_POLYTRAUMA getNewPatient(int? id, TBI_POLYTRAUMA polyTrauma)
        {
            if (id.HasValue)
            {

                using (_db = GetDataContext())
                {
                    //_db.DeferredLoadingEnabled = false;

                    //polyTrauma.SynchroniseWithDataContext(_db);

                    polyTrauma.PATIENT = (from se in _db.PATIENTs where se.PATIENT_ID == id select se).FirstOrDefault();
                }
            }

            return polyTrauma;

        }

        #endregion

        #region EnforceBusinessRules

        public int EnforceBusinessRules(ref TBI_POLYTRAUMA polyTrauma)
        {
            using (_db = GetDataContext())
            {
                int status = polyTrauma.EnforceBusinessRules(false, _db);
            }

            return 0;
        }


        #endregion
    }
}
